/*
 * Routines for dealing with Agents
 */

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_fms_comm.h"
#include "lf_fabric.h"
#include "lf_alert.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fma.h"
#include "fms_notify.h"
#include "fms_fabric.h"
#include "fms_fabric_delta.h"
#include "fms_switch.h"
#include "fms_alert.h"

/*
 * local prototypes
 */
void fms_fma_changed_link(struct lf_nic *nicp, int port,
                          struct fma_fms_nic *nip);
void fms_fma_new_link(struct lf_nic *nicp, int port,
                          struct fma_fms_nic *nip);
void fms_fma_missing_link(struct lf_nic *nicp, int port);

/*
 * Reconcile reported link info with database and generate alerts and
 * adjust fabric for differences where appropriate.
 */
void
fms_reconcile_link_info(
  struct lf_host *hp,
  struct fma_fms_host_msg *hmp)
{
  struct lf_fabric *fp;
  int num_active_ports;
  int inn;
  int n;

  fp = F.fabvars->fabric;		/* get pointer to fabric */

  inn = ntohl(hmp->nic_cnt_32);		/* info # of NICs */

  /* Check that each NIC is connected where it should be */
  for (n=0; n<inn; ++n) {
    struct fma_fms_nic *nip;
    struct lf_nic *nicp;
    int nic_id;
    int i;

    /* use nic_id to get nic info pointer */
    nip = hmp->nic_array + n;
    nic_id = ntohs(nip->nic_id_16);
    nicp = lf_find_nic_by_id(hp, nic_id);

    num_active_ports = ntohl(nip->num_active_ports_32);

    /* check connection for each port */
    for (i=0; i<num_active_ports; ++i) {
      struct lf_xbar *xp;
      int fma_xbar_id;
      int fma_xbar_port;

      /* This is what the FMA reports itself as attached to */
      fma_xbar_id = ntohl(nip->xbar_id_32[i]);
      fma_xbar_port = ntohl(nip->xbar_port_32[i]);

      /* get the xbar for this connection */
      xp = LF_XBAR(nicp->topo_ports[i]);

      /* nothing recorded as attached in map */
      if (xp == NULL) {

	/* -1 for ID means nothing found, so all is as expected */
	if (fma_xbar_id == -1) {
	  fms_nic_link_down(nicp, i);
	  continue;

	/* something in xbar_id means a new connection for us */
	} else {
	  fms_fma_new_link_by_xid(nicp, i, fma_xbar_id, fma_xbar_port);
	  fms_nic_link_up(nicp, i);
	}

      /* something is there in the map, verify it */
      } else {

	/* needsto be an xbar */
	if (xp->ln_type != LF_NODE_XBAR) {
	  LF_ERROR(("non-xbar host connection not supported"));
	}

	/* nothing found on host, mark the link "down" */
	if (fma_xbar_id == -1) {
	  fms_fma_missing_link(nicp, i);
	  fms_nic_link_down(nicp, i);

	} else {

	  /* If xbar has valid xid, check for a match */
	  if (xp->xbar_id != 0) {

	    /* If xids non-zero and match and ports match, all is good */
	    if (xp->xbar_id == fma_xbar_id
		&& nicp->topo_rports[i] == fma_xbar_port) {
	      fms_notify(FMS_EVENT_DEBUG, "%s:%d:%d attached as expected",
		           nicp->host->hostname, nicp->host_nic_id, i);
	      fms_nic_link_up(nicp, i);
	      continue;

	    /* xbar info doesn't match, this link changed... */
	    } else {
	      fms_notify(FMS_EVENT_INFO, "%s says xbar %d:%d, DB says %d:%d",
	                   hp->hostname,
			   fma_xbar_id, fma_xbar_port,
			   xp->xbar_id, nicp->topo_rports[i]);
	      fms_fma_changed_link(nicp, i, nip);
	      fms_nic_link_up(nicp, i);
	    }

	  /* xbar does not have an ID, other means may be necessary */
	  } else {

	    /* if FMA has non-zero xbar, it has moved to one with an ID */
	    if (fma_xbar_id != 0) {
	      fms_notify(FMS_EVENT_INFO, "%s says xbar %d:%d, DB says %d:%d",
	                   hp->hostname,
			   fma_xbar_id, fma_xbar_port,
			   xp->xbar_id, nicp->topo_rports[i]);
	      fms_fma_changed_link(nicp, i, nip);
	      fms_nic_link_up(nicp, i);

	    /* correctly connected to a mystery xbar - mark link up */
	    } else {
	      fms_nic_link_up(nicp, i);

	      /* XXX ought to do something more to validate */
	      continue;
	    }
	  }
	}
      }
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * A NIC link has changed
 */
void
fms_fma_changed_link(
  struct lf_nic *nicp,
  int port,
  struct fma_fms_nic *nip)
{
  printf("XXX changed_link\n");

  /* remove the the old link */
  /* fms_remove_nic_link(nicp, port); */

  /* add the new link */
  /* XXX */
}

/*
 * A NIC link is new
 */
void
fms_fma_new_link_by_xid(
  struct lf_nic *nicp,
  int port,
  int xbar_id,
  int xbar_port)
{
  struct lf_xbar *xp;

  /* find this xbar in fabric */
  xp = lf_find_xbar_by_id(F.fabvars->fabric, xbar_id);
  if (xp == NULL) {
  }

  fms_notify(FMS_EVENT_DEBUG, "XXX new_link_by_xid");
}


/*
 * A NIC link is missing
 */
void
fms_fma_missing_link(
  struct lf_nic *nicp,
  int port)
{
  printf("XXX missing_link\n");
}

/*
 * Remove a fabric link - this unlinks the nodes and updates the link struct
 * in the FMS private data.
 */
void
fms_remove_topo_link(
  union lf_node *np,
  int port)
{
  struct fms_link *linkp;

  /* remove this link from the database */
  fms_remove_topo_link_from_db(np, port);

  /* get to struct for this link */
  linkp = fms_get_node_link(np, port);
	      
  /* clear any alerts associated with this link */
  fms_cancel_subject_alerts(linkp->fl_alerts_anchor);

  /* reduce reference count on link struct */
  fms_dereference_link(linkp);
  fms_set_node_link(np, port, NULL);

  /* this unlinks the nodes from each other */
  lf_remove_link_by_topo(np, port);
}

/*
 * Make a note that a link is being removed.  Not just down - actually gone.
 */
void
fms_note_link_removed(
  union lf_node *np,
  int port)
{
  printf("XXX link being removed\n");
}

/*
 * Note than a NIC port has been oberved to be up.  If the port is
 * just "DOWN" mark it as up, but don't change it if in other states, like
 * "MAINTENANCE".
 */
void
fms_nic_link_up(
  struct lf_nic *nicp,
  int port)
{
  struct fms_link *linkp;

  /* set the state */
  fms_set_node_link_state(LF_NODE(nicp), port, LF_LINK_STATE_UP);

  /* If link currently down, rescind any alerts */
  if (nicp->link_state[port] == LF_LINK_STATE_DOWN) {

    /* reset both possible types of alert */
    linkp = FMS_NIC(nicp)->nic_port_data[port]->link;
    fms_switch_alert_link_up(linkp);
    fms_fma_alert_nic_port_up(nicp, port);
  }
}

/*
 * Note than a NIC port has been oberved to be down.
 * If the port was thought to be UP, then
 *   if there is a connection, alert that the link is down
 *   if there is no connection, alert that the port is down
 */
void
fms_nic_link_down(
  struct lf_nic *nicp,
  int port)
{
  struct fms_link *linkp;
  union lf_node *np;

  /* If link currently marked up or unknown, mark it down. */
  if (nicp->link_state[port] == LF_LINK_STATE_UNKNOWN ||
      nicp->link_state[port] == LF_LINK_STATE_UP) {
    fms_set_node_link_state(LF_NODE(nicp), port, LF_LINK_STATE_DOWN);

    /* If xbar connection, alert for the link */
    np = nicp->topo_ports[port];
    if (np != NULL && np->ln_type == LF_NODE_XBAR) {
      linkp = FMS_NIC(nicp)->nic_port_data[port]->link;
      fms_switch_alert_xbar_host_link_down(LF_XBAR(np),
					   nicp->topo_rports[port]);

    /* else just an alert on the NIC */
    } else {
      fms_fma_alert_nic_port_down(nicp, port);
    }
  }
}
